/*===========================================================*/
/* GPS miniPOI            		                     */
/* (c) MSP			                     	     */
/*===========================================================*/

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this package; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
 

#include <avr/pgmspace.h>
#include "common.h"
#include "gps.h"
#include "serial.h"
#include "poi.h"
#include "sound.h"


//***********************************************************
// Global variables definition
//***********************************************************

uint8_t PROGMEM pkGPGGA[] = "$GPGGA";
uint8_t PROGMEM pkGPVTG[] = "$GPVTG";
uint8_t PROGMEM pkPROG[] = "$PROG";

uint8_t	volatile	gps_con_cnt;

// pointer to bootloader
void (*bootloader_start)(void) = (void*)0xFC00;

/*------------------------------------------------*/
/* Interrupt handlers                             */
/*------------------------------------------------*/

// Timer 0 Output Compare mach
// frequency 500 Hz (period 2 ms)
SIGNAL (SIG_OUTPUT_COMPARE0)
{
   static uint8_t 	cnt5 = 0;
   static uint8_t 	cnt10 = 0;
   static uint8_t	led_cnt = 0;
   
   TIMSK &= ~(1 << OCIE0); 					// disable tmr 0 output compare interrupt
   OCR0 += 250;
   SEI();							// enable global interrupts

   if (++cnt5 == 5)						// 100Hz (10 ms)
   {
      cnt5 = 0;

      sound_tick();
      
      if (++cnt10 == 10)					// 10Hz (100ms)
      {
	 cnt10 = 0;

	 // led blink counter
	 if (++led_cnt == 10)
	    led_cnt = 0;

	 // GPS connection timeout
	 if (gps_con_cnt != 20)
	    gps_con_cnt++;

	 if (   ((gps_con_cnt == 20) || (GPSinfo.satnum == 0)) 
	     && (led_cnt < 4))
	    PORTE &= ~LED_GR;
	 else
	    PORTE |= LED_GR;

      }


   }

   TIMSK |= (1 << OCIE0);					// enable tmr 0 output compare interrupt
}

    

/*------------------------------------------------*/
/* Main program                                   */
/*------------------------------------------------*/

int main(void)
{
   unsigned int		i;
   unsigned int		j;
   
   //stop errant interrupts until set up
   CLI(); 							// disable all interrupts

   
   // Initialise ports
   PORTA = 0x00;
   DDRA  = 0x00;
   PORTB = 0x00;
   DDRB  = SPK;
   PORTC = 0x00;
   DDRC  = 0x00;
   PORTD = 0x00;
   DDRD  = 0x00;
   PORTE = 0x00;
   DDRE  = LED_RD | LED_GR;


   // GLOBAL initialisation
   MCUCR = 0x00;
   MCUCSR = 0x00;
      
   SEI(); 							// enable interrupts

   // Initialise serial port
   com_init();

  
   // Initialise timer 0
   // 16 MHz / 128 = 125KHz -> 8 us/tick
   TCCR0 = 0x00; 						// stop
   OCR0  = 250;							// set output compare; 125KHz/250=500Hz
   TCCR0 = (1 << CS02) | (1 << CS00);				// prescaler 128
   

   // timer interrupt sources
   TIMSK = (1 << OCIE0);					// output capture int enabled


    
   gps_con_cnt = 0;

   // main loop
   while (1)
   {
      if (rx_buf_ready != 0)
      {
	 if (!strncmp_P(rx_buf, pkGPGGA, 6))
	 {
            gps_con_cnt = 0;					// clear timeout indicator
	    gps_process_GGA(rx_buf);				// process GGA string
	    poi(GPSinfo.longitude, GPSinfo.lattitude);		// process POI

	 } else if (!strncmp_P(rx_buf, pkGPVTG, 6))
	 {
            gps_con_cnt = 0;					// clear timeout indicator
	    gps_process_VTG(rx_buf);				// process VTG string

	 } else if (!strncmp_P(rx_buf, pkPROG, 5))
	 {
	    // enter bootloader
	    CLI();
	    bootloader_start();					// start bootloader

	 }

	 rx_buf_ready = 0;					// clear RX buffer
      }
   }

}

