// =====================================================
//
//   AVR 910 Programmer routines
//
//   (c) MSP 2007
//
// =====================================================

// -----------------------------------------------------
//  Written with help of AvrAtmel.c,
//  part of UISP by Uros Platise
// -----------------------------------------------------

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this package; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

#include <stdlib.h>
#include <stdio.h>


#include "common.h"
#include "serial.h"
#include "avr.h"

#define PAGESIZE	256

uint8_t tx_buffer[PAGESIZE];
uint8_t tx_buffer_empty;
uint16_t tx_buffer_ptr;
uint32_t tx_buffer_saddr;

uint8_t rx_buffer[PAGESIZE];

uint8_t device;

// ------------------------------------------------------------------------

/*------------------------------------------------*/
/* Verify response of the bootloader              */
/*------------------------------------------------*/
void avr_CheckResponse(uint8_t data[], uint8_t length)
{
   uint8_t i;

   ser_recbuffer(rx_buffer, length);
   for (i=0; i<length; i++)
      if (rx_buffer[i] != data[i])
      {
         fprintf(stderr, "Device not responding correctly. %s vs %s\n", rx_buffer, data );
	 exit(1);
      }
}
   
/*------------------------------------------------*/
/* Enter programming mode                         */
/*------------------------------------------------*/
void avr_EnterProgrammingMode()
{
   // identify programmer
   BYTE get_id[4] = {0x1B, 0x1B, 0x1B, 'S'};
   ser_sendbuffer(get_id, 4);
   avr_CheckResponse((uint8_t*)"AVRBOOT", 7);

   // get device
   BYTE get_device[1] = {'t'};
   ser_sendbuffer(get_device, 1);
   ser_recbuffer(&device, 1);
   avr_CheckResponse((uint8_t*)"\x00", 1);

   // set device
   BYTE set_device[2] = {'T', device};
   ser_sendbuffer(set_device, 2);
   avr_CheckResponse((uint8_t*)"\x0D", 1);


   // prepare buffer
   tx_buffer_empty = !0;
   for (tx_buffer_ptr = 0; tx_buffer_ptr < 255; tx_buffer_ptr++)
      tx_buffer[tx_buffer_ptr] = 0xFF;
   tx_buffer_ptr = 0;
   tx_buffer_saddr = 0;
}

/*------------------------------------------------*/
/* Leave programming mode                         */
/*------------------------------------------------*/
void avr_LeaveProgrammingMode()
{
   avr_FlushWriteBuffer();
   
   BYTE leave_prog[1] = {'E'};
   ser_sendbuffer(leave_prog, 1);
   //avr_CheckResponse((uint8_t*)"\x0D", 1); FIXME
}


/*------------------------------------------------*/
/* Set address                                    */
/*------------------------------------------------*/
void avr_SetAddress(uint16_t addr)
{
   uint8_t setAddr[3] = { 'A', (addr>>8)&0xff, addr&0xff};
   ser_sendbuffer(setAddr, 3);
   avr_CheckResponse((uint8_t*)"\x0D", 1);
}



/*------------------------------------------------*/
/* Program memory page                            */
/*------------------------------------------------*/
void avr_WriteProgramMemoryPage()
{
   avr_SetAddress(tx_buffer_saddr>>1);
   uint8_t setBuf[4] = {'B', (PAGESIZE&0xFF00)>>8, PAGESIZE&0xFF, 'F'};
   ser_sendbuffer(setBuf, 4);
   ser_sendbuffer(tx_buffer, PAGESIZE);
   avr_CheckResponse((uint8_t*)"\x0D", 1);
}


/*------------------------------------------------*/
/* Flush write buffer                             */
/*------------------------------------------------*/
void avr_FlushWriteBuffer()
{
  if (!tx_buffer_empty)
  {
    avr_WriteProgramMemoryPage();  
    tx_buffer_empty = !0;
  }
}


/*------------------------------------------------*/
/* Write byte to the memory                       */
/*------------------------------------------------*/
// TODO addr is byte-address, add check for page start on even address   
void avr_WriteByte(uint8_t segment, uint32_t addr, uint8_t data)
{
   if (segment==SEG_FLASH)
   {
      if (addr > (131071 - 1024))
      {
	 fprintf(stderr, "Address out of range!\n" );
	 exit(1);
      }

      if (tx_buffer_empty && (data == 0xFF))
	 return;

      if (tx_buffer_empty)
      {
	 tx_buffer_empty = 0;
	 tx_buffer_saddr = addr;
	 tx_buffer_ptr = 0;
	 tx_buffer[tx_buffer_ptr++] = data;
      } else
      {
	 if ((addr-tx_buffer_saddr) > (PAGESIZE-1))
	 {
	    avr_WriteProgramMemoryPage();
	    tx_buffer_saddr = addr;
	    for (tx_buffer_ptr = 0; tx_buffer_ptr < PAGESIZE; tx_buffer_ptr++)
	       tx_buffer[tx_buffer_ptr] = 0xFF;
	    tx_buffer_ptr = 0;
	 }
	 tx_buffer[tx_buffer_ptr++] = data;
      }


   } else if (segment==SEG_EEPROM)
   {
      avr_SetAddress(addr);
      uint8_t writeEE[5] = {'B', 0, 1, 'E', data};
      ser_sendbuffer(writeEE, 5);
      avr_CheckResponse((uint8_t*)"\x0D", 1);
   }

}





