/*===========================================================*/
/*							     */
/* Serial Port Routines			                     */
/*							     */
/* (c) MSP 2007						     */
/*							     */
/*===========================================================*/

// -----------------------------------------------------------
//  Based on AVR911 - AVR Open-source Programmer
// -----------------------------------------------------------

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this package; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

#include <stdlib.h>
#include <stdio.h>

#include <time.h>
#include "serial.h"


long ser_portNumber; 			// COMx port number.
long ser_timeout; 			// Desired timeout limit when receiving data.
HANDLE ser_handle; 			// Win32 device handle for the com port.
COMMTIMEOUTS ser_oldComTimeouts; 	// Store old serial port timeout parameters.
unsigned char ser_channelOpen; 		// Is channel open?

       
/*------------------------------------------------*/
/* Serial port initialization                     */
/*------------------------------------------------*/
void ser_init (long portnumber, long speed, long timeout)
{
   char comName[] = "COMx";
   COMMTIMEOUTS ser_comTimeouts;
   DCB dcb;


   /* Check if channel already open */
   if( ser_channelOpen )
   {
     fprintf(stderr, "Channel already open! Cannot open port twice.\n" );
     exit(1);
   }

   /* Generate COM filename and attempt open */
   comName[3] = '0' + portnumber;
   ser_handle = CreateFile( comName, GENERIC_READ | GENERIC_WRITE, 0, NULL,
	        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

   /* Print error and return if failed opening port */
   if( ser_handle == INVALID_HANDLE_VALUE )
   {
      fprintf(stderr, "Error opening COM port!\n" );
      exit(1);
   }

   ser_channelOpen = true;

   /* Store old COM port settings */
   if( !GetCommTimeouts( ser_handle, &ser_oldComTimeouts ) )
   {
      fprintf(stderr, "Error reading COM port settings!\n" );
      exit(1);
   }

   /* Get another copy of the COM settings, and change them */
   if( !GetCommTimeouts( ser_handle, &ser_comTimeouts ) )
   {
      fprintf(stderr, "Error reading COM port settings\n!" );
      exit(1);
   }

   ser_comTimeouts.ReadIntervalTimeout = MAXDWORD;
   ser_comTimeouts.ReadTotalTimeoutConstant = 0;
   ser_comTimeouts.ReadTotalTimeoutMultiplier = 0;

   /* Apply new settings */
   if( !SetCommTimeouts( ser_handle, &ser_comTimeouts ) )
   {
      fprintf(stderr, "Error changing COM port settings!\n" );
      exit(1);
   }

   ser_timeout = timeout;


   /* Get another copy of the COM settings, and change them */
   if( !GetCommState( ser_handle, &dcb ) )
   {
      fprintf(stderr, "Error reading COM port settings2!\n" );
      exit(1);
   }

   dcb.BaudRate = speed;
   dcb.ByteSize = 8;
   dcb.Parity = NOPARITY;
   dcb.StopBits = ONESTOPBIT;
   dcb.fAbortOnError = TRUE;

   // set XON/XOFF
   dcb.fOutX = FALSE;                    		// XON/XOFF off for transmit
   dcb.fInX = FALSE;                    		// XON/XOFF off for receive
   // set RTSCTS
   dcb.fOutxCtsFlow = FALSE;                    	// turn on CTS flow control
   dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;    	//
   // set DSRDTR
   dcb.fOutxDsrFlow = FALSE;                    	// turn off DSR flow control
   dcb.fDtrControl = DTR_CONTROL_DISABLE;    		//

   /* Apply new settings */
   if( !SetCommState( ser_handle, &dcb ) )
   {
      fprintf(stderr, "Error changing COM port settings2!\n" );
      exit(1);
   }

}



/*------------------------------------------------*/
/* Close serial port                              */
/*------------------------------------------------*/
void ser_close()
{
   if(!ser_channelOpen )
      return;

   /* Restore old COM parameters */
   if( !SetCommTimeouts( ser_handle, &ser_oldComTimeouts ) )
   {
      fprintf(stderr, "Error changing COM port settings!\n" );
      exit(1);
   }

   /* Release port */
   if( ser_handle != INVALID_HANDLE_VALUE )
      if( !CloseHandle( ser_handle ) )
      {
         fprintf(stderr, "Error closing COM port!\n");
	 exit(1);
      }

   ser_channelOpen = false;
}


/*------------------------------------------------*/
/* Send byte                                      */
/*------------------------------------------------*/
void ser_sendbyte( long data )
{
   DWORD written;

   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot send to unopened channel.\n" );
      exit(1);
   }

   /* Attempt writing */
   if( !WriteFile( ser_handle, &data, 1, &written, NULL ) )
   {
      fprintf(stderr, "Error writing byte to COM port!\n" );
      exit(1);
   }

}


/*------------------------------------------------*/
/* Receive byte                                   */
/*------------------------------------------------*/
long ser_getbyte()
{
   time_t startTime;
   startTime = time( NULL ); // Read current time in seconds
   DWORD readnum;
   unsigned char data;

   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot read from unopened channel.\n" );
      exit(1);
   }

   /* Attempt receiving byte until timeout limit exceeded */
   do
   {
      /* Read byte from port */
      if( !ReadFile( ser_handle, &data, 1, &readnum, NULL ) )
      {
         fprintf(stderr, "Error reading byte from COM port!\n" );
	 exit(1);
      }

      if( readnum == 1 )
         return ((long) data) & 0xff;

   } while( time(NULL) - startTime < ser_timeout );

   /* Timeout */
   fprintf(stderr, "Timeout during COM-port read operation!\n" );
   exit(1);
}


/*------------------------------------------------*/
/* Clear TX buffer                                */
/*------------------------------------------------*/
void ser_flushTX()
{
   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot flush an unopened channel.\n" );
      exit(1);
   }

   /* Purge data from write buffer */
   if( !PurgeComm( ser_handle, PURGE_TXCLEAR ) )
   {
      fprintf(stderr, "Error flushing COM port TX buffer!\n" );
      exit(1);
   }
}



/*------------------------------------------------*/
/* Clear RX buffer                                */
/*------------------------------------------------*/
void ser_flushRX()
{
   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot flush an unopened channel.\n" );
      exit(1);
   }

   /* Purge data from write buffer */
   if( !PurgeComm( ser_handle, PURGE_RXCLEAR ) )
   {
      fprintf(stderr, "Error flushing COM port RX buffer!\n" );
      exit(1);
   }
}



/*------------------------------------------------*/
/* Send multiple bytes                            */
/*------------------------------------------------*/
void ser_sendbuffer( unsigned char *data, DWORD bufsize )
{
   DWORD written;

   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot write to unopened channel.\n" );
      exit(1);
   }

   /* Attempt writing */
   if( !WriteFile( ser_handle, data, bufsize, &written, NULL ) )
   {
      fprintf(stderr, "Error writing multiple bytes to COM port!\n" );
      exit(1);
   }
}


/*------------------------------------------------*/
/* Receive multiple bytes                         */
/*------------------------------------------------*/
DWORD ser_recbuffer( unsigned char *p_data, DWORD bufsize )
{
   DWORD total_len=0;
   DWORD rem, len;
   time_t startTime;
   startTime = time(NULL); // Read current time in seconds

   /* Check if channel is open */
   if( !ser_channelOpen )
   {
      fprintf(stderr, "Channel not open! Cannot write to unopened channel.\n" );
      exit(1);
   }


   while(total_len<bufsize)
   {
      rem = bufsize-total_len;

      if( !ReadFile( ser_handle, p_data+total_len, rem, &len, NULL ) )
      {
         fprintf(stderr, "Error reading byte from COM port!\n" );
	 exit(1);
      }
      
      total_len += len;

      if (time(NULL) - startTime > ser_timeout)
      {
         fprintf(stderr, "Timeout when reading data from COM port!\n" );
	 exit(1);
      }

   }
   return total_len;
}
   
 
